/***************************************************************************
 *
 * Copyright (c) 2014 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include <stdint.h>
#include <cmath>
#include <utility>
#include <algorithm>
#include <cfloat>
#include <iterator>
#include <limits>
#include <string>
#include <sstream>
#include "Matrix.h"

using namespace std;
using namespace LayerManagerCalibration;

void printMatrixData(Matrix::Data& matrixData, ostream& stream = cout)
{
    for (int i = 0; i < MATRIX_ROWS; ++i)
    {
        for (int j = 0; j < MATRIX_COLUMNS; ++j)
        {
            if (j != (MATRIX_COLUMNS - 1))
                stream << matrixData[i][j] << ", ";
            else
                stream << matrixData[i][j];
        }
        stream << endl;
    }

    stream << endl;
}

bool checkEquivalence(Matrix& a, Matrix& b)
{
    bool matricesEqual = true;

    for (int i = 0; i < MATRIX_ROWS; ++i)
    {
        for (int j = 0; j < MATRIX_COLUMNS; ++j)
        {
            // Adapted from http://www.cygnus-software.com/papers/comparingfloats/comparingfloats.htm
            // Check if the numbers are really close -- needed
            // when comparing numbers near zero.
            double diff = fabs(a.getData()[i][j] - b.getData()[i][j]);
            if (diff <= numeric_limits<double>::epsilon())
                continue;

            double A = fabs(a.getData()[i][j]);
            double B = fabs(b.getData()[i][j]);
            double largest = (B > A) ? B : A;

            if (!(diff <= largest * numeric_limits<double>::epsilon()))
                matricesEqual = false;
        }
    }

    return matricesEqual;
}

bool checkEquivalence(Matrix& matrixA, Matrix::Data& matrixData)
{
    bool matricesEqual = true;

    for (int i = 0; i < MATRIX_ROWS; ++i)
    {
        for (int j = 0; j < MATRIX_COLUMNS; ++j)
        {
            // Adapted from http://www.cygnus-software.com/papers/comparingfloats/comparingfloats.htm
            // Check if the numbers are really close -- needed
            // when comparing numbers near zero.
            double diff = fabs(matrixA.getData()[i][j] - matrixData[i][j]);
            if (diff <= numeric_limits<double>::epsilon())
                continue;

            double A = fabs(matrixA.getData()[i][j]);
            double B = fabs(matrixData[i][j]);
            double largest = (B > A) ? B : A;

            if (!(diff <= largest * numeric_limits<double>::epsilon()))
                matricesEqual = false;
        }
    }

    return matricesEqual;
}

void displayTestResult(std::string& testPoint, bool result)
{
    //Test equivalence
    if (!result)
    {
        cerr << "TEST FAILURE - " << testPoint << endl;
        cerr << endl;
    }
    else
    {
        cout << "TEST PASS - " << testPoint << endl;
        cout << endl;
    }
}

bool testMatrixInitialisation()
{
    Matrix::Data matrixData = {{1, 2, 3},{3, 2, 1},{0, 3, -1}};
    bool result = false;
    string testName("Create a Matrix from an array structure");

    // Test initialisation
    cout << testName << ":" << endl;
    Matrix testMatrix(matrixData);
    printMatrixData(matrixData);

    // Test equivalence
    result = checkEquivalence(testMatrix, matrixData);

    displayTestResult(testName, result);

    return result;
}

bool testMatrixExtraction()
{
    Matrix::Data matrixData = {{20.5, 1, 8},{1, 30, 7},{5, 2, -100}};
    bool result = false;
    string testName("Extract output stream from Matrix");
    ostringstream streamFromData, streamFromMatrix;

    // Test initialisation
    cout << testName << ":" << endl;
    Matrix testMatrix(matrixData);
    cout << "Initialising Array" << endl;
    printMatrixData(matrixData, streamFromData);
    cout << endl;
    cout << "Output from created Matrix" << endl;
    streamFromMatrix << testMatrix << endl;
    cout << endl;

    // Ask user if matrix is correctly formed
    result = (streamFromData.str() == streamFromMatrix.str());

    cout << "streamFromData: '" << streamFromData.str() << "'" << endl;
    cout << "streamFromMatrix: '" << streamFromMatrix.str() << "'" << endl;

    displayTestResult(testName, result);

    return result;
}

bool testMatrixCopyConstructor()
{
    Matrix::Data matrixData = {{1, 2, 3},{3, 2, 1},{0, 3, -1}};
    bool result = false;
    string testName("Create a Matrix by copying another array");

    Matrix testMatrix(matrixData);

    // Test copy construction
    cout << testName << ":" << endl;
    Matrix testMatrixTwo(testMatrix);
    cout << testMatrixTwo << endl;

    // Test equivalence
    result = checkEquivalence(testMatrix, testMatrixTwo);

    displayTestResult(testName, result);

    return result;
}

bool testMatrixDefaultConstructor()
{
    Matrix testMatrix;
    bool result = false;
    string testName("Create a new Matrix through default constructor");

    // Create an equivalent default explicitly
    Matrix::Data matrixData = {{0, 0, 0},{0, 0, 0},{0, 0, 0}};
    Matrix testMatrixB(matrixData);

    cout << testName << ":" << endl;
    cout << testMatrix << endl;

    // Test equivalence
    result = checkEquivalence(testMatrix, testMatrixB);

    displayTestResult(testName, result);

    return result;
}

bool testMatrixSetFromArrayStruct()
{
    Matrix::Data matrixData = {{1, 2, 3},{3, 2, 1},{0, 3, -1}};
    Matrix testMatrix;
    bool result = false;
    string testName("Test Set a Matrix by equating to an array structure");

    cout << testName << endl;
    testMatrix = matrixData;
    cout << testMatrix << endl;

    // Test equivalence
    result = checkEquivalence(testMatrix, matrixData);

    displayTestResult(testName, result);

    return result;
}

bool testMatrixTransposition()
{
    Matrix::Data matrixData = {{1, 2, 3},{3, 2, 1},{0, 3, -1}};
    Matrix::Data transposedMatrixData = {{1, 3, 0},{2, 2, 3},{3, 1, -1}};
    string testName("Test Matrix Transposition");
    bool result = false;
    Matrix testMatrix(matrixData);
    Matrix testMatrixTranspose(transposedMatrixData);

    cout << testName << ":" << endl;
    cout << "Matrix Before Transposition:" << endl;
    cout << testMatrix << endl;

    // Test Transposition of matrix
    testMatrix.transposeMatrix();

    cout << "Matrix After Transposition:" << endl;
    cout << testMatrix << endl;

    // Test equivalence
    result = checkEquivalence(testMatrix, testMatrixTranspose);

    displayTestResult(testName, result);

    return result;
}

bool testMatrixAdjugate()
{
    Matrix::Data matrixData = {{8,7,7},{6,9,2},{-6,9,-2}};
    Matrix::Data matrixDataAdj = {{-36, 77, -49}, {0, 26, 26}, {108, -114, 30}};
    bool result = false;
    string testName("Test Adjugate a Matrix");
    Matrix testMatrix(matrixData);
    Matrix testMatrixAdjugate(matrixDataAdj);

    cout << testName << ":" << endl;
    cout << "Matrix Before Adjugate:" << endl;
    cout << testMatrix << endl;

    // Test Adjugate Matrix
    testMatrix.adjugateMatrix();

    cout << "Matrix After Adjugate:" << endl;
    cout << testMatrix << endl;

    // Test equivalence
    result = checkEquivalence(testMatrix, testMatrixAdjugate);

    displayTestResult(testName, result);

    return result;
}

bool testMatrixDeterm()
{
    Matrix::Data matrixData = {{9, 3, 5}, {-6, -9, 7}, {-1, -8, 1}};
    bool result = false;
    string testName("Test Determinant of Matrix");
    Matrix testMatrix(matrixData);

    // Test Matrix Determinate
    double determ = testMatrix.calculateDeterminant();

    cout << testName << ":" << endl << testMatrix << endl;
    cout << "is: " << determ << endl;

    // Test equivalence with calculated value
    result = fabs(determ - 615.0) < numeric_limits<double>::epsilon();

    displayTestResult(testName, result);

    return result;
}

bool testMatrixbyMatrixMultiplication()
{
    Matrix::Data matrixDataOne = {{1, 2, 3},{3, 2, 1},{0, 3, -1}};
    Matrix::Data matrixDataTwo = {{4, 5, 6},{1, 7, 0},{5, 1, 8}};
    Matrix::Data matrixDataExpectedResult = {{21, 22, 30},{19, 30, 26},{-2, 20, -8}};
    bool result = false;
    string testName("Test Matrix by Matrix Multiplication");
    Matrix matrixOne(matrixDataOne);
    Matrix matrixTwo(matrixDataTwo);
    Matrix matrixExpectedResult(matrixDataExpectedResult);

    // Test Matrix Multiplication
    cout << testName << ":" << std::endl;
    cout << "Matrix 1:" << endl;
    cout << matrixOne << endl;
    cout << "Matrix 2:" << endl;
    cout << matrixTwo << endl;
    cout << "Multiply matrix: " << endl;

    // Perform calculation
    Matrix matrixCalculatedResult = matrixOne * matrixTwo;

    // Display result
    cout << matrixCalculatedResult << endl;

    // Test equivalence
    result = checkEquivalence(matrixCalculatedResult, matrixExpectedResult);

    displayTestResult(testName, result);

    return result;

}

bool testMatrixbyFactorMultiplication()
{
    Matrix::Data matrixData = {{9,4,1}, {-6, -9, 3}, {-1, -6, 9}};
    Matrix::Data expectedMatrixData = {{4.5,2,0.5}, {-3, -4.5, 1.5}, {-0.5, -3, 4.5}};
    bool result = false;
    string testName("Test Matrix by Factor Multiplication");
    Matrix testMatrix(matrixData);
    Matrix testExpectedResult(expectedMatrixData);

    // Test Matrix Multiplication
    cout << testName << ":" << endl;
    cout << "Matrix:" << endl;
    cout << testMatrix << endl;
    cout << "Factor: 0.5" << endl;

    Matrix calcultedResult = testMatrix * 0.5;

    cout << "Resulting Matrix:" << endl;
    cout << calcultedResult << endl;

    // Test equivalence
    result = checkEquivalence(calcultedResult, testExpectedResult);

    displayTestResult(testName, result);

    if (!result)
    {
        cout << "Expected: " << endl;
        cout << testExpectedResult << endl;
    }

    return result;
}

bool testMatrixEquivalenceOperator()
{
    Matrix::Data matrixData = {{9, 3, 5}, {-6, -9, 7}, {-1, -8, 1}};
    bool result = false;
    string testName("Test Matrix Equivalence Operator");
    Matrix testMatrixOne(matrixData);
    Matrix testMatrixTwo(matrixData);

    cout << testName << ":" << endl;
    cout << "Matrix 1:" << endl;
    cout << testMatrixOne << endl;
    cout << "Matrix 2:" << endl;
    cout << testMatrixTwo << endl;

    // Test equivalence using class method
    result = (testMatrixOne == testMatrixTwo);

    displayTestResult(testName, result);

    return result;
}

bool testMatrixStreamBehaviour()
{
    bool result = false;
    string testName("Test Initialise Matrix from stream input and output operation");
    LayerManagerCalibration::Matrix matrixIn;
    char input;
    istringstream testInput("1 2 3 4 5 6 7 8 9");
    string testOutput = "1, 2, 3\n4, 5, 6\n7, 8, 9\n\n";
    ostringstream oss;

    cout << endl;
    cout << testName << ":" << endl;

    // Test Input From Stream
    testInput >> matrixIn;
    // Display whats been input
    oss << matrixIn << endl;

    result = (oss.str() == testOutput);

    displayTestResult(testName, result);

    return result;
}

int main( int argc, const char* argv[] )
{
    bool result = true;

    result = testMatrixInitialisation() && result;

    result = testMatrixExtraction() && result;

    result = testMatrixCopyConstructor() && result;

    result = testMatrixDefaultConstructor() && result;

    result = testMatrixSetFromArrayStruct() && result;

    result = testMatrixTransposition() && result;

    result = testMatrixAdjugate() && result;

    result = testMatrixDeterm() && result;

    result = testMatrixbyMatrixMultiplication() && result;

    result = testMatrixbyFactorMultiplication() && result;

    result = testMatrixEquivalenceOperator() && result;

    result = testMatrixStreamBehaviour() && result;

    if (result)
    {
        cout << "ALL MATRIX TESTS PASSED" << endl;
        return 0;
    }
    else
    {
        cerr << "1 OR MORE MATRIX TESTS FAILED - TEST FAILURE" << endl;
        return 1;
    }
}
